﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Tilemaps;

//LSystemDrawer represents a logical drawer for applying the output of an L-System contextually into the game world
public class LSystemDrawer : MonoBehaviour
{
    //Public properties
    public int StartingLength = 8;
    public int LengthDecrement = 2;
    public int LengthResetValue = 2;

    //The Drawer computes the city boundaries based on its output
    public Vector3Int CityTopLeftTilePosition { get; private set; }
    public Vector3Int CityBottomLeftTilePosition { get; private set; }
    public Vector3Int CityTopRightTilePosition { get; private set; }
    public Vector3Int CityBottomRightTilePosition { get; private set; }
    public Vector3Int CityCentreTilePosition { get; private set; }
    public Vector2Int CitySize { get; private set; }
    public List<Vector3Int> RoadPositions { get; private set; } = new List<Vector3Int>();

    //Protected common properties for output
    protected Vector3Int _CurrentGridCellPosition;
    protected Vector2Int _CurrentDirection;
    protected int _CurrentLength;
    protected Stack<DrawParameters> _SavedParameters;

    //Local private variables for computing city data
    private static Tile _TransparentTile;
    private Vector3Int _LeftMostTilePosition;
    private Vector3Int _RightMostTilePosition;
    private Vector3Int _TopMostTilePosition;
    private Vector3Int _BottomMostTilePosition;
    private Dictionary<Vector3Int, GameObject> _RoadGameObjects = new Dictionary<Vector3Int, GameObject>();

    /// <summary>
    /// Base function that SHOULD be overridden to draw the sentence as an output in the city
    /// </summary>
    /// <param name="sentenceToDraw">The sentence to use for drawing</param>
    /// <returns>Was the drawing successful or not?</returns>
    public virtual bool Draw(string sentenceToDraw) 
    {
        //Initialize our variables
        _SavedParameters = new Stack<DrawParameters>();
        _TransparentTile = Resources.Load("Palette Tiles/transparent") as Tile;

        //Compute our positions
        _LeftMostTilePosition = new Vector3Int(GameManager.Instance.RoadsTilemap.size.x / 2, GameManager.Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);
        _RightMostTilePosition = new Vector3Int(GameManager.Instance.RoadsTilemap.size.x / 2, GameManager.Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);
        _TopMostTilePosition = new Vector3Int(GameManager.Instance.RoadsTilemap.size.x / 2, GameManager.Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);
        _BottomMostTilePosition = new Vector3Int(GameManager.Instance.RoadsTilemap.size.x / 2, GameManager.Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);

        return false; 
    }

    /// <summary>
    /// Finalizes the roads by setting the correct junction models
    /// </summary>
    protected void FinalizeRoads()
    {
        //Create objects for each potential road section
        Object horizontalObject = Resources.Load("Models/Roads/horRoad", typeof(GameObject));
        Object verticalObject = Resources.Load("Models/Roads/verRoad", typeof(GameObject));
        Object topLeftCornerObject = Resources.Load("Models/Roads/topLeftCornerRoad", typeof(GameObject));
        Object topRightCornerObject = Resources.Load("Models/Roads/topRightCornerRoad", typeof(GameObject));
        Object bottomLeftCornerObject = Resources.Load("Models/Roads/bottomLeftCornerRoad", typeof(GameObject));
        Object bottomRightCornerObject = Resources.Load("Models/Roads/bottomRightCornerRoad", typeof(GameObject));
        Object topJunctionObject = Resources.Load("Models/Roads/topJunctionRoad", typeof(GameObject));
        Object bottomJunctionObject = Resources.Load("Models/Roads/bottomJunctionRoad", typeof(GameObject));
        Object leftJunctionObject = Resources.Load("Models/Roads/leftJunctionRoad", typeof(GameObject));
        Object rightJunctionObject = Resources.Load("Models/Roads/rightJunctionRoad", typeof(GameObject));
        Object centreJunctionObject = Resources.Load("Models/Roads/centreJunctionRoad", typeof(GameObject));

        //Let's set the centre tile to have the centre junction as a sentinel value - this is so it can run in the loop and get its correct texture set rather than being grass
        Vector3Int centreTilePos = new Vector3Int(GameManager.Instance.RoadsTilemap.size.x / 2, GameManager.Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);
        DrawRoadAndSetTile(centreTilePos, Instantiate(leftJunctionObject) as GameObject, Constants.LeftJunctionRoadTileID);

        //Loop through all the tiles on the roads later
        for (int x = GameManager.Instance.RoadsTilemap.origin.x; x < GameManager.Instance.RoadsTilemap.origin.x + GameManager.Instance.RoadsTilemap.size.x; x++)
        {
            for (int y = GameManager.Instance.RoadsTilemap.origin.y; y < GameManager.Instance.RoadsTilemap.origin.y + GameManager.Instance.RoadsTilemap.size.y; y++)
            {
                Vector3Int thisTilePosition = new Vector3Int(x, y, Constants.TilemapZPosition);
                TileBase thisTile = GameManager.Instance.RoadsTilemap.GetTile(thisTilePosition);

                if (thisTile != null && ((Tile)thisTile).name.Contains("Road"))
                {
                    //It's a road, so get it's neighbours
                    TileBase topNeighbour = null;
                    TileBase bottomNeighbour = null;
                    TileBase leftNeighbour = null;
                    TileBase rightNeighbour = null;

                    if (x >= GameManager.Instance.RoadsTilemap.origin.x)
                    {
                        leftNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(thisTilePosition.x - 1, thisTilePosition.y, thisTilePosition.z));

                        if (leftNeighbour != null && !((Tile)leftNeighbour).name.Contains("Road"))
                        {
                            leftNeighbour = null;
                        }
                    }

                    if (x < GameManager.Instance.RoadsTilemap.origin.x + GameManager.Instance.RoadsTilemap.size.x)
                    {
                        rightNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(thisTilePosition.x + 1, thisTilePosition.y, thisTilePosition.z));

                        if (rightNeighbour != null && !((Tile)rightNeighbour).name.Contains("Road"))
                        {
                            rightNeighbour = null;
                        }
                    }

                    if (y >= GameManager.Instance.RoadsTilemap.origin.y)
                    {
                        topNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(thisTilePosition.x, thisTilePosition.y + 1, thisTilePosition.z));

                        if (topNeighbour != null && !((Tile)topNeighbour).name.Contains("Road"))
                        {
                            topNeighbour = null;
                        }
                    }

                    if (y < GameManager.Instance.RoadsTilemap.origin.y + GameManager.Instance.RoadsTilemap.size.y)
                    {
                        bottomNeighbour = GameManager.Instance.RoadsTilemap.GetTile(new Vector3Int(thisTilePosition.x, thisTilePosition.y - 1, thisTilePosition.z));

                        if (bottomNeighbour != null && !((Tile)bottomNeighbour).name.Contains("Road"))
                        {
                            bottomNeighbour = null;
                        }
                    }

                    //Got the neighbours, draw the correct road depending on the neighbours
                    if (leftNeighbour != null && rightNeighbour != null && topNeighbour == null && bottomNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, horizontalObject, Constants.HorizontalRoadTileID);
                    }

                    else if (leftNeighbour == null && rightNeighbour == null && topNeighbour != null && bottomNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, verticalObject, Constants.VerticalRoadTileID);
                    }

                    else if (rightNeighbour == null && bottomNeighbour == null && leftNeighbour != null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, horizontalObject, Constants.HorizontalRoadTileID);
                        _RoadGameObjects[thisTilePosition].FindChild("RoadDeadEndRight").SetActive(true);
                    }

                    else if (rightNeighbour != null && bottomNeighbour == null && leftNeighbour == null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, horizontalObject, Constants.HorizontalRoadTileID);
                        _RoadGameObjects[thisTilePosition].FindChild("RoadDeadEndLeft").SetActive(true);
                    }

                    else if (rightNeighbour == null && bottomNeighbour != null && leftNeighbour == null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, verticalObject, Constants.VerticalRoadTileID);
                        _RoadGameObjects[thisTilePosition].FindChild("RoadDeadEndTop").SetActive(true);
                    }

                    else if (rightNeighbour == null && bottomNeighbour == null && leftNeighbour == null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, verticalObject, Constants.VerticalRoadTileID);
                        _RoadGameObjects[thisTilePosition].FindChild("RoadDeadEndBottom").SetActive(true);
                    }

                    else if (rightNeighbour != null && bottomNeighbour != null && leftNeighbour == null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, topLeftCornerObject, Constants.TopLeftCornerRoadTileID);
                    }

                    else if (rightNeighbour == null && bottomNeighbour != null && leftNeighbour != null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, topRightCornerObject, Constants.TopRightCornerRoadTileID);
                    }

                    else if (rightNeighbour != null && bottomNeighbour == null && leftNeighbour == null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, bottomLeftCornerObject, Constants.BottomLeftCornerRoadTileID);
                    }

                    else if (rightNeighbour == null && bottomNeighbour == null && leftNeighbour != null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, bottomRightCornerObject, Constants.BottomRightCornerRoadTileID);
                    }

                    else if (rightNeighbour != null && bottomNeighbour != null && leftNeighbour != null && topNeighbour == null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, topJunctionObject, Constants.TopJunctionRoadTileID);
                    }

                    else if (rightNeighbour != null && bottomNeighbour == null && leftNeighbour != null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, bottomJunctionObject, Constants.BottomJunctionRoadTileID);
                    }

                    else if (rightNeighbour != null && bottomNeighbour != null && leftNeighbour == null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, leftJunctionObject, Constants.LeftJunctionRoadTileID);
                    }

                    else if (rightNeighbour == null && bottomNeighbour != null && leftNeighbour != null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, rightJunctionObject, Constants.RightJunctionRoadTileID);
                    }

                    else if (rightNeighbour != null && bottomNeighbour != null && leftNeighbour != null && topNeighbour != null)
                    {
                        DrawRoadAndSetTile(thisTilePosition, centreJunctionObject, Constants.CentreJunctionRoadTileID);
                    }
                }
            }
        }

        //All done drawing, we can compute our boundaries now
        CityTopLeftTilePosition = new Vector3Int(_LeftMostTilePosition.x, _TopMostTilePosition.y, _LeftMostTilePosition.z);
        CityBottomLeftTilePosition = new Vector3Int(_LeftMostTilePosition.x, _BottomMostTilePosition.y, _LeftMostTilePosition.z);
        CityTopRightTilePosition = new Vector3Int(_RightMostTilePosition.x, _TopMostTilePosition.y, _RightMostTilePosition.z);
        CityBottomRightTilePosition = new Vector3Int(_RightMostTilePosition.x, _BottomMostTilePosition.y, _RightMostTilePosition.z);
        CitySize = new Vector2Int(CityTopRightTilePosition.x - CityTopLeftTilePosition.x, CityTopLeftTilePosition.y - CityBottomLeftTilePosition.y);
        CityCentreTilePosition = new Vector3Int(CityTopLeftTilePosition.x + (CitySize.x / 2), CityBottomLeftTilePosition.y + (CitySize.y / 2), CityTopLeftTilePosition.z);

        _RoadGameObjects.Clear();   //We don't need the info any more, let's free up memory
    }

    /// <summary>
    /// Draws a road model at the specified position, updating the tilemap data to reflect it
    /// </summary>
    /// <param name="cellPosition">The position to draw it</param>
    /// <param name="roadObject">The road model to draw</param>
    /// <param name="tileName">The name to set the occupied tile</param>
    private void DrawRoadAndSetTile(Vector3Int cellPosition, Object roadObject, string tileName)
    {
        //Let's make the tile itself transparent
        Tile thisTransparentTile = Instantiate(_TransparentTile) as Tile;
        thisTransparentTile.name = tileName;

        GameManager.Instance.RoadsTilemap.SetTile(cellPosition, thisTransparentTile);
        GameManager.Instance.RoadsTilemap.GetTile(cellPosition).name = tileName;

        //Remove duplicates
        if (_RoadGameObjects.ContainsKey(cellPosition))
        {
            Destroy(_RoadGameObjects[cellPosition]);
            _RoadGameObjects.Remove(cellPosition);
        }

        //Spawn the road on the corrected world position with the correct data
        GameObject roadGameObject = Instantiate(roadObject) as GameObject;
        roadGameObject.transform.parent = GameManager.Instance.RoadsTilemap.transform;
        roadGameObject.transform.position = Utilities.GetModelWorldPosition(cellPosition, roadGameObject.GetComponent<MeshFilter>().sharedMesh, roadGameObject.transform.eulerAngles.z);
        roadGameObject.name = tileName;
        roadGameObject.layer = LayerMask.NameToLayer("Roads");
        roadGameObject.SetActive(ConfigurationManager.Instance.Core.DrawRoadModels);
        _RoadGameObjects[cellPosition] = roadGameObject;
        RoadPositions.Add(cellPosition);

        //Update the minimap as well
        MinimapManager.Instance.DrawMinimapTile(cellPosition, ColoursManager.Instance.Colours["MinimapRoad"].Colour);

        //Finally, update our boundary computation tile data if appropriate
        if(cellPosition.x < _LeftMostTilePosition.x)
        {
            _LeftMostTilePosition = cellPosition;
        }

        else if (cellPosition.x > _RightMostTilePosition.x)
        {
            _RightMostTilePosition = cellPosition;
        }

        else if(cellPosition.y > _TopMostTilePosition.y)
        {
            _TopMostTilePosition = cellPosition;
        }

        else if(cellPosition.y < _BottomMostTilePosition.y)
        {
            _BottomMostTilePosition = cellPosition;
        }
    }

    /// <summary>
    /// Draws an initial horizontal or vertical road (will be corrected later with junctions when FinalizeRoads is called)
    /// </summary>
    /// <param name="cellPosition">The position of the cell to draw a road on</param>
    /// <param name="horizontalRoad">Is the road horizontal?</param>
    protected void DrawInitialRoad(Vector3Int cellPosition, bool horizontalRoad = true)
    {
        if(horizontalRoad)
        {
            DrawRoadAndSetTile(cellPosition, Resources.Load("Models/Roads/horRoad"), Constants.HorizontalRoadTileID);
        }

        else
        {
            DrawRoadAndSetTile(cellPosition, Resources.Load("Models/Roads/verRoad"), Constants.VerticalRoadTileID);
        }
    }
}